/*
 *  Crestron
 * UiHAL interface file for touch data
 *
 * Ray Kuschan
 */


#include <linux/ft5x06_i2c_defs.h>
#include "../ft5x06_ts_i2c.h"

#include <linux/proc_fs.h>  /* Necessary because we use proc fs */
#include <asm/uaccess.h>    /* for copy_*_user */

#define TOUCH_DIR_NAME           "touch"
#define TOUCH_MT_FILENAME           "MTevent"  //Multi touch
#define PROCFS_MAX_SIZE     1024

#define MAX_TOUCH_POINTS_SUPPORTED 2
#define PEN_DOWN        1
#define PEN_UP          0


/** * The structure keeping information about the /proc file */
struct proc_dir_entry *TouchDir;
struct proc_dir_entry *MTeventFile;

/** * The buffer (2k) for this module */
char gMTevent_buffer[PROCFS_MAX_SIZE];

/** * The size of the data held in the buffer */
unsigned long gMTevent_buffer_size = 0;

/* maintain touch ID */
unsigned long gTouchID[MAX_TOUCH_POINTS_SUPPORTED] = {0,0};

struct UiHalTouch uiTouchData[MAX_TOUCH_POINTS_SUPPORTED];
int uiMultiTouch=0;
struct ft5x06* uiTS;

extern void processTouchData(struct ft5x06 *ts, int touchId, u8 eventId, u16 X, u16 Y);


static ssize_t procfs_read(struct file *filp,  /* see include/linux/fs.h   */
                             char *buffer,      /* buffer to fill with data */
                             size_t length,     /* length of the buffer     */
                             loff_t * offset)
{
  static int finished = 0;
  char * p = gMTevent_buffer;

  /* needed to stop from continuously printing */
  if ( finished == 1 ) { finished=0; return 0; }
  finished = 1;

   p += sprintf ( p, "TOUCH_ID_1=%d\n" , uiTouchData[0].touchId );
   p += sprintf ( p, "PRESS_STATE_1=%d\n" , uiTouchData[0].pressState );
   p += sprintf ( p, "X_POSITION_1=%d\n" , uiTouchData[0].xPosition );
   p += sprintf ( p, "Y_POSITION_1=%d\n" , uiTouchData[0].yPosition );

   p += sprintf ( p, "TOUCH_ID_2=%d\n" , uiTouchData[1].touchId );
   p += sprintf ( p, "PRESS_STATE_2=%d\n" , uiTouchData[1].pressState );
   p += sprintf ( p, "X_POSITION_2=%d\n" , uiTouchData[1].xPosition );
   p += sprintf ( p, "Y_POSITION_2=%d\n" , uiTouchData[1].yPosition );

   p += sprintf ( p, "NUM_TOUCHES_SUPPORTED=%d\n" , MAX_TOUCH_POINTS_SUPPORTED );
   p += sprintf ( p, "FIRST_ID=%d\n" , 0 );
   gMTevent_buffer_size = p-gMTevent_buffer;

        /*
         * We use put_to_user to copy the string from the kernel's
         * memory segment to the memory segment of the process
         * that called us. get_from_user, BTW, is
         * used for the reverse.
         */
        if ( copy_to_user(buffer, gMTevent_buffer, gMTevent_buffer_size) ) {
                return -EFAULT;
        }

  return gMTevent_buffer_size;

}

static ssize_t procfs_write(struct file *filp,  /* see include/linux/fs.h   */
                             char *buffer,      /* buffer to fill with data */
                             size_t length,     /* length of the buffer     */
                             loff_t * offset)
{
	char* tag = NULL;
	char* value = NULL;
	char** tempPtr = &buffer;
	struct UiHalTouch FakeTouchData;

	gMTevent_buffer_size = length;
	if (gMTevent_buffer_size > PROCFS_MAX_SIZE ) {
		gMTevent_buffer_size = PROCFS_MAX_SIZE;
	}

	if ( copy_from_user(gMTevent_buffer, buffer, gMTevent_buffer_size) )
	{
		return -EFAULT;
	}
	tag = strsep ( tempPtr, "=" );

	if ( strcmp ( tag, "FAKETOUCH" ) == 0 )
	{

		value = strsep ( tempPtr, "=" );
		sscanf ( value, "%d,%d,%d,%d", &FakeTouchData.touchId, &FakeTouchData.pressState, &FakeTouchData.xPosition, &FakeTouchData.yPosition);

		if(FakeTouchData.touchId >= MAX_TOUCH_POINTS_SUPPORTED)
		{
			printk(KERN_ERR "Unsupported touch ID\n");
			return -EINVAL;
		}
		if(!(FakeTouchData.pressState == FT_NTCH || FakeTouchData.pressState == FT_TCH))
		{
			printk(KERN_ERR "Unsupported touch state\n");
			return -EINVAL;
		}

		processTouchData(uiTS, FakeTouchData.touchId, FakeTouchData.pressState, FakeTouchData.xPosition, FakeTouchData.yPosition);
		input_sync(uiTS->input);
	}

	return gMTevent_buffer_size;
}

static void init_uiTouchData(void)
{
        int i;

        for (i=0; i<MAX_TOUCH_POINTS_SUPPORTED;i++)
        {
                uiTouchData[i].touchId  =i;
                uiTouchData[i].pressState       =0;
                uiTouchData[i].xPosition        =0;
                uiTouchData[i].yPosition        =0;
        }
}
/*
 * This function decides whether to allow an operation
 * (return zero) or not allow it (return a non-zero
 * which indicates why it is not allowed).
 *
 * The operation can be one of the following values:
 * 0 - Execute (run the "file" - meaningless in our case)
 * 2 - Write (input to the kernel module)
 * 4 - Read (output from the kernel module)
 *
 * This is the real function that checks file
 * permissions. The permissions returned by ls -l are
 * for referece only, and can be overridden here.
 */

static int module_permission(struct inode *inode, int op, struct nameidata *foo)
{
//  if ( op == 2 ) // no writes
//  {
//    return -EACCES;
//  }

  return 0;
}


/*
 * The file is opened - we don't really care about
 * that, but it does mean we need to increment the
 * module's reference count.
 */
int open_procfs(struct inode *inode, struct file *file)
{
        try_module_get(THIS_MODULE);
        return 0;
}

/*
 * The file is closed - again, interesting only because
 * of the reference count.
 */
int close_procfs(struct inode *inode, struct file *file)
{
        module_put(THIS_MODULE);
        return 0;               /* success */
}

static struct file_operations File_Ops_File = {
        .read    = procfs_read,
        .write   = procfs_write,
        .open    = open_procfs,
        .release = close_procfs,
};

/*
 * Inode operations for our proc file. We need it so
 * we'll have some place to specify the file operations
 * structure we want to use, and the function we use for
 * permissions. It's also possible to specify functions
 * to be called for anything else which could be done to
 * an inode (although we don't bother, we just put
 * NULL).
 */

static struct inode_operations Inode_Ops_File = {
        .permission = module_permission,        /* check for permissions */
};

int uiTouchIntfInit(struct ft5x06* ts)
{
	int error;

	uiTS = ts;

 /* create the UIHAL /proc file for touch data */
    TouchDir = proc_mkdir(TOUCH_DIR_NAME,  NULL);
    if (TouchDir == NULL){
            printk(KERN_ALERT "Error: Could not initialize /proc/%s\n",
                   TOUCH_DIR_NAME);
            return -ENOMEM;
    }
    else
    {

        MTeventFile = create_proc_entry(TOUCH_MT_FILENAME, 0644, TouchDir);
          if (MTeventFile == NULL){
              printk(KERN_ALERT "Error: Could not initialize /proc/%s/%s\n",
                     TOUCH_DIR_NAME,TOUCH_MT_FILENAME);
              return -ENOMEM;
         }
          else
          {
          //MTeventFile->owner = THIS_MODULE;
          MTeventFile->proc_iops = &Inode_Ops_File;
          MTeventFile->proc_fops = &File_Ops_File;
          MTeventFile->mode = S_IFREG | S_IRUGO | S_IWUSR;
          MTeventFile->uid = 0;
          MTeventFile->gid = 0;
          MTeventFile->size = 80;
        }
    }

        return error;
}



